/** LineFeature.java in the package org.RCSImitate.sensoryItems of the RCSImitate project.
    Originally created 7-Nov-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */
package org.RCSImitate.sensoryItems;

/** Represents a line in a RoboCup game
 * 
 * @author Michael W. Floyd
 * @since 0.2
 */
public class LineFeature extends RoboCupSimulation2DSpatialObject{
	
	private static final long serialVersionUID = 7215504570812781693L;

	public static String c_LINE = "RoboCupSimulation.Line";
	
	public static String c_LOCATION_UNKNOWN = "UNKNOWN";
	public static String c_LOCATION_LEFT = "l";
	public static String c_LOCATION_RIGHT = "r";
	public static String c_LOCATION_TOP = "t";
	public static String c_LOCATION_BOTTOM = "b";
	
	private String m_location;
	
	/** Creates a line with an unknown location.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public LineFeature(){
		super(LineFeature.c_LINE);
		this.m_location = LineFeature.c_LOCATION_UNKNOWN;
	}
	
	/** Sets the location of the line. Acceptable locations are
	 * 'l' (left), 'r' (right), 't' (top) and 'b' (bottom).
	 * 
	 * @param s The location of the line
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public void setLocation(String s){
		//check parameters
		if(s == null){
			throw new IllegalArgumentException("Null parameter given.");
		}
		//check for invalid locations
		String lowerside = s.toLowerCase();
		if( !(lowerside.equals(LineFeature.c_LOCATION_LEFT)) && !(lowerside.equals(LineFeature.c_LOCATION_RIGHT)) && !(lowerside.equals(LineFeature.c_LOCATION_TOP)) && !(lowerside.equals(LineFeature.c_LOCATION_BOTTOM))){
			throw new IllegalArgumentException("Illegal line position: " + lowerside);
		}
		
		this.m_location = lowerside;
	}
	
	/** Return the location of the line
	 * 
	 * @return The location of the line
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public String getLocation(){
		return new String(this.m_location);
	}
	
	/** Overrides the equals method
	 * 
	 */
	@Override
	public boolean equals(Object o){
		//check for same object
		if(this == o){
			return true;
		}
		//check for null or wrong class type
		if(o == null || !(o instanceof LineFeature)){
			return false;
		}
		
		//typecast
		LineFeature lf = (LineFeature)o;
		
		//check the super class attributes
		if(!super.equals(lf)){
			return false;
		}
		
		return this.m_location.equals(lf.getLocation());
	}

}
