/** PlayerFeature.java in the package org.RCSImitate.sensoryItems of the RCSImitate project.
    Originally created 7-Nov-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */
package org.RCSImitate.sensoryItems;

/** A general representation of a player in RoboCup Simulation soccer
 * 
 * @author Michael W. Floyd
 * @since 0.2
 */
public class PlayerFeature extends RoboCupSimulation2DSpatialObject {

	private static final long serialVersionUID = -2584431565607613947L;

	public static final int c_UNIFORM_DEFAULT = -1;
	
	private int m_uniformNumber;
	private boolean m_isGoalie;
	private float m_headDirection;
	private float m_bodyDirection;
	
	/** Creates a PlayerFeature with the given feature name.
	 * The body direction and head direction are both set
	 * to 0.0 by default, the uniform number is set to a
	 * default value and the player is not a goalie.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public PlayerFeature(String featureName){
		super(featureName);
		
		//set the default values
		this.m_bodyDirection = 0.0f;
		this.m_headDirection = 0.0f;
		this.m_uniformNumber = PlayerFeature.c_UNIFORM_DEFAULT;
		this.m_isGoalie = false;
		
	}
	
	/** Returns the body direction of the player
	 * 
	 * @return body direction of player
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public float getBodyDirection() {
		return this.m_bodyDirection;
	}

	/** Sets the body direction of the player
	 * 
	 * @param direction The body direction
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public void setBodyDirection(float direction) {
		this.m_bodyDirection = direction;
	}

	/** Gets the head direction of the player
	 * 
	 * @return head direction
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public float getHeadDirection() {
		return this.m_headDirection;
	}

	/** Set the head direction of the player
	 * 
	 * @param direction The head direction
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public void setHeadDirection(float direction) {
		this.m_headDirection = direction;
	}

	/** Returns true if the player is a goalie.
	 * 
	 * @return True if player is a goalie
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public boolean isGoalie() {
		return this.m_isGoalie;
	}

	/** Sets if the player is a goalie.
	 * 
	 * @param goalie True if the player is a goalie
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public void setIsGoalie(boolean goalie) {
		this.m_isGoalie = goalie;
	}

	/** Returns the uniform number of the player.
	 * 
	 * @return The uniform number
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public int getUniformNumber() {
		return this.m_uniformNumber;
	}

	/** Sets the uniform number of the player
	 * 
	 * @param number The uniform number
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	public void setUniformNumber(int number) {
		this.m_uniformNumber = number;
	}
	
	/** Overrides the equals method
	 * 
	 */
	@Override
	public boolean equals(Object o){
		//check for exact same object
		if(this == o){
			return true;
		}
		
		//check for null or different class
		if(o == null || !(o instanceof PlayerFeature)){
			return false;
		}
		
		//typecast the object
		PlayerFeature pf = (PlayerFeature)o;
		
		//check the super class attributes
		if(!super.equals(pf)){
			return false;
		}
		
		return ( (this.m_bodyDirection == pf.getBodyDirection()) && 
				 (this.m_headDirection == pf.getHeadDirection()) && 
				 (this.m_isGoalie == pf.isGoalie()) && 
				 (this.m_uniformNumber == pf.getUniformNumber()));
	}
}
