/** LogFile2CaseBaseTest.java in the package tests.junit.org.RCSImitate.casebasebuilder of the RCSImitate project.
    Originally created 7-Nov-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package tests.junit.org.RCSImitate.casebasebuilder;


import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.List;

import org.RCSImitate.casebasebuilder.LogFile2CaseBase;
import org.RCSImitate.sensoryItems.BallFeature;
import org.JIFSA.AgentInputs;
import org.JIFSA.Case;
import org.JIFSA.CaseBase;
import org.JIFSA.SensoryItem;
import org.JIFSA.tools.CaseBaseIO;
import org.junit.Test;

/** Tests the org.RCSImitate.casebasebuilder.LogFile2CaseBase class
 * 
 * @author Michael W. Floyd
 * @since 0.2 
 */
public class LogFile2CaseBaseTest {

	private final String c_logFile = "tests\\testdata\\testlogfile.log";
	private final String c_caseFile = "tests\\testdata\\testlogfile.cb";
	private final String c_teamName = "Canada";
	
	/** Test to make sure the constructor acts
	 * as expected when given a null parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor_nullLogFile(){
		new LogFile2CaseBase(null,this.c_caseFile,this.c_teamName);
	}
	
	/** Test to make sure the constructor acts
	 * as expected when given a null parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor_nullCaseFile(){
		new LogFile2CaseBase(this.c_logFile,null,this.c_teamName);
	}
	
	/** Test to make sure the constructor acts
	 * as expected when given a null parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor_nullTeamName(){
		new LogFile2CaseBase(this.c_logFile,this.c_caseFile,null);
	}
	
	/** Test to see if an exception is thrown when
	 * a file that does not exist is given
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IOException.class)
	public void parseLogFile_logFileDoesNotExist() throws Exception{
		LogFile2CaseBase lf2cb = new LogFile2CaseBase("doesnotexist.log",this.c_caseFile,this.c_teamName);
		lf2cb.parseLogFile();
	}
	
	/** Tests to see if a case is created for every "see" message
	 * in the log file.
	 * 
	 *
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void casebaseEqualsSeeMessages() throws Exception{
		LogFile2CaseBase lf2cb = new LogFile2CaseBase(this.c_logFile,this.c_caseFile,this.c_teamName);
		lf2cb.parseLogFile();
		CaseBase cb = lf2cb.getCaseBase();
		
		BufferedReader in = new BufferedReader(new FileReader(this.c_logFile));
		
		int seeCounter = 0;
		String nextMessage = null;
		while ((nextMessage = in.readLine()) != null){
			if(nextMessage.indexOf("see") != -1){
				seeCounter ++;
			}
		}
		
		in.close();
		
		assertEquals(cb.getCasebaseSize(),seeCounter);
	}
	
	/** Make sure that it adds all the actions that exist in the 
	 * log file.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void actionsEqualActionMessages() throws Exception {
		LogFile2CaseBase lf2cb = new LogFile2CaseBase(this.c_logFile,this.c_caseFile,this.c_teamName);
		lf2cb.parseLogFile();
		
		CaseBase cb = lf2cb.getCaseBase();
		List<Case> caselist = cb.getCaseList();
		int actionCounter = 0;
		for(Case currCase: caselist){
			actionCounter += currCase.getActions().size();
		}
		
		BufferedReader in = new BufferedReader(new FileReader(this.c_logFile));
		
		int messageCounter = 0;
		String nextMessage = null;
		while ((nextMessage = in.readLine()) != null){
			if(nextMessage.startsWith("(kick")){
				messageCounter ++;
			}else if(nextMessage.startsWith("(dash")){
				messageCounter ++;
			}else if(nextMessage.startsWith("(turn")){
				messageCounter ++;
			}else if(nextMessage.startsWith("(turn_neck")){
				messageCounter ++;
			}else if(nextMessage.startsWith("(catch")){
				messageCounter ++;
			}
		}
		
		in.close();
		
		assertEquals(actionCounter,messageCounter);
	}
	

	/** Tests to see that the case base is properly written
	 * to the output file
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void casebaseWritenProperly() throws Exception{
		LogFile2CaseBase lf2cb = new LogFile2CaseBase(this.c_logFile,this.c_caseFile,this.c_teamName);
		lf2cb.parseLogFile();
		CaseBase cb = lf2cb.getCaseBase();
		lf2cb.writeCaseBaseFile();
		
		CaseBase read = CaseBaseIO.loadCaseBase(this.c_caseFile);
		
		assertTrue(cb.equals(read));
		
	}
	
	/** Tests to make sure editing the case base that is returned
	 * does not modify the original
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void getCaseBase_modify() throws Exception{
		LogFile2CaseBase lf2cb = new LogFile2CaseBase(this.c_logFile,this.c_caseFile,this.c_teamName);
		lf2cb.parseLogFile();
		CaseBase cb = lf2cb.getCaseBase();
		int originalSize = cb.getCasebaseSize();
		
		Case c = new Case();
		cb.addCase(c);
		
		cb = lf2cb.getCaseBase();
		assertEquals(cb.getCasebaseSize(),originalSize);	
	}
	
	/** Make sure it properly handles a null message
	 * parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IllegalArgumentException.class)
	public void parseSeeMessage_nullMessage(){
		String s = "myTeam";
		LogFile2CaseBase.parseSeeMessage(null, s);	
	}
	
	/** Make sure it properly handles a null team name
	 * parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IllegalArgumentException.class)
	public void parseSeeMessage_nullTeamName(){
		String s = "(see)";
		LogFile2CaseBase.parseSeeMessage(s, null);
	}
	
	/** Tests to see if an invalid see message is handled
	 * correctly.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IllegalArgumentException.class)
	public void parseSeeMessage_invalidSeeMessage(){
		String s = "not a see message";
		LogFile2CaseBase.parseSeeMessage(s, "myTeam");
	}
	
	/** Tests a see message with Ball and no distance/direction
	 * changes
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void parseSeeMessage_Ball_noChange(){
		String see1 = "(see 1 ((Ball) 22 13))";
		AgentInputs av = LogFile2CaseBase.parseSeeMessage(see1, "myTeam");
		assertEquals(av.getNumberSensoryItems(),1);
		assertEquals(av.getNumberSensoryItems(BallFeature.c_BALL),1);
		List<SensoryItem> sensoryItems = av.getSensoryItems(BallFeature.c_BALL);
		assertEquals(sensoryItems.size(),1);
		SensoryItem f = sensoryItems.get(0);
		assertTrue(f instanceof BallFeature);
		BallFeature bf = (BallFeature)f;
		assertEquals(bf.getDirection(), 13f);
		assertEquals(bf.getDirectionChange(), 0f);
		assertEquals(bf.getDistance(), 22f);
		assertEquals(bf.getDistanceChange(), 0f);
	}
	
	/** Tests a see message with Ball and distance/direction
	 * changes
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void parseSeeMessage_Ball_change(){
		String see1 = "(see 1 ((Ball) 22 13 0.54 -7))";
		AgentInputs av = LogFile2CaseBase.parseSeeMessage(see1, "myTeam");
		assertEquals(av.getNumberSensoryItems(),1);
		assertEquals(av.getNumberSensoryItems(BallFeature.c_BALL),1);
		List<SensoryItem> sensoryItems = av.getSensoryItems(BallFeature.c_BALL);
		assertEquals(sensoryItems.size(),1);
		SensoryItem f = sensoryItems.get(0);
		assertTrue(f instanceof BallFeature);
		BallFeature bf = (BallFeature)f;
		assertEquals(bf.getDirection(), 13f);
		assertEquals(bf.getDirectionChange(), -7f);
		assertEquals(bf.getDistance(), 22f);
		assertEquals(bf.getDistanceChange(), 0.54f);
	}
	
	/** Tests a see message with B and no distance/direction
	 * changes
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void parseSeeMessage_B_noChange(){
		String see1 = "(see 13 ((B) 98 -1))";
		AgentInputs av = LogFile2CaseBase.parseSeeMessage(see1, "myTeam");
		assertEquals(av.getNumberSensoryItems(),1);
		assertEquals(av.getNumberSensoryItems(BallFeature.c_BALL),1);
		List<SensoryItem> features = av.getSensoryItems(BallFeature.c_BALL);
		assertEquals(features.size(),1);
		SensoryItem f = features.get(0);
		assertTrue(f instanceof BallFeature);
		BallFeature bf = (BallFeature)f;
		assertEquals(bf.getDirection(), -1f);
		assertEquals(bf.getDirectionChange(), 0f);
		assertEquals(bf.getDistance(), 98f);
		assertEquals(bf.getDistanceChange(), 0f);
	}
	
	/** Tests a see message with B and distance/direction
	 * changes
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void parseSeeMessage_B_change(){
		String see1 = "(see 13 ((B) 98 -1 -14 9))";
		AgentInputs av = LogFile2CaseBase.parseSeeMessage(see1, "myTeam");
		assertEquals(av.getNumberSensoryItems(),1);
		assertEquals(av.getNumberSensoryItems(BallFeature.c_BALL),1);
		List<SensoryItem> features = av.getSensoryItems(BallFeature.c_BALL);
		assertEquals(features.size(),1);
		SensoryItem f = features.get(0);
		assertTrue(f instanceof BallFeature);
		BallFeature bf = (BallFeature)f;
		assertEquals(bf.getDirection(), -1f);
		assertEquals(bf.getDirectionChange(), 9f);
		assertEquals(bf.getDistance(), 98f);
		assertEquals(bf.getDistanceChange(), -14f);
	}
	
	/** Tests a see message with B and no distance/direction
	 * changes
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void parseSeeMessage_b_noChange(){
		String see1 = "(see 13 ((b) 98 -1))";
		AgentInputs av = LogFile2CaseBase.parseSeeMessage(see1, "myTeam");
		assertEquals(av.getNumberSensoryItems(),1);
		assertEquals(av.getNumberSensoryItems(BallFeature.c_BALL),1);
		List<SensoryItem> features = av.getSensoryItems(BallFeature.c_BALL);
		assertEquals(features.size(),1);
		SensoryItem f = features.get(0);
		assertTrue(f instanceof BallFeature);
		BallFeature bf = (BallFeature)f;
		assertEquals(bf.getDirection(), -1f);
		assertEquals(bf.getDirectionChange(), 0f);
		assertEquals(bf.getDistance(), 98f);
		assertEquals(bf.getDistanceChange(), 0f);
	}
	
	/** Tests a see message with b and distance/direction
	 * changes
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void parseSeeMessage_b_change(){
		String see1 = "(see 13 ((b) 98 -1 -14 9))";
		AgentInputs av = LogFile2CaseBase.parseSeeMessage(see1, "myTeam");
		assertEquals(av.getNumberSensoryItems(),1);
		assertEquals(av.getNumberSensoryItems(BallFeature.c_BALL),1);
		List<SensoryItem> features = av.getSensoryItems(BallFeature.c_BALL);
		assertEquals(features.size(),1);
		SensoryItem f = features.get(0);
		assertTrue(f instanceof BallFeature);
		BallFeature bf = (BallFeature)f;
		assertEquals(bf.getDirection(), -1f);
		assertEquals(bf.getDirectionChange(), 9f);
		assertEquals(bf.getDistance(), 98f);
		assertEquals(bf.getDistanceChange(), -14f);
	}
	
	/** Tests a see message with ball and no distance/direction
	 * changes
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void parseSeeMessage_ball_noChange(){
		String see1 = "(see 13 ((ball) 98 -1))";
		AgentInputs av = LogFile2CaseBase.parseSeeMessage(see1, "myTeam");
		assertEquals(av.getNumberSensoryItems(),1);
		assertEquals(av.getNumberSensoryItems(BallFeature.c_BALL),1);
		List<SensoryItem> features = av.getSensoryItems(BallFeature.c_BALL);
		assertEquals(features.size(),1);
		SensoryItem f = features.get(0);
		assertTrue(f instanceof BallFeature);
		BallFeature bf = (BallFeature)f;
		assertEquals(bf.getDirection(), -1f);
		assertEquals(bf.getDirectionChange(), 0f);
		assertEquals(bf.getDistance(), 98f);
		assertEquals(bf.getDistanceChange(), 0f);
	}
	
	/** Tests a see message with ball and distance/direction
	 * changes
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void parseSeeMessage_ball_change(){
		String see1 = "(see 13 ((ball) 98 -1 -14 9))";
		AgentInputs av = LogFile2CaseBase.parseSeeMessage(see1, "myTeam");
		assertEquals(av.getNumberSensoryItems(),1);
		assertEquals(av.getNumberSensoryItems(BallFeature.c_BALL),1);
		List<SensoryItem> features = av.getSensoryItems(BallFeature.c_BALL);
		assertEquals(features.size(),1);
		SensoryItem f = features.get(0);
		assertTrue(f instanceof BallFeature);
		BallFeature bf = (BallFeature)f;
		assertEquals(bf.getDirection(), -1f);
		assertEquals(bf.getDirectionChange(), 9f);
		assertEquals(bf.getDistance(), 98f);
		assertEquals(bf.getDistanceChange(), -14f);
	}
	
	/** Tests a see message with multiple ball objects
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void parseSeeMessage_multipleBalls(){
		String see1 = "(see 13 ((ball) 98 -1 -14 9) ((ball) 5 0.01) ((ball) 9 14 9.7 -2))";
		AgentInputs av = LogFile2CaseBase.parseSeeMessage(see1, "myTeam");
		assertEquals(av.getNumberSensoryItems(),3);
		assertEquals(av.getNumberSensoryItems(BallFeature.c_BALL),3);
		List<SensoryItem> features = av.getSensoryItems(BallFeature.c_BALL);
		assertEquals(features.size(),3);
		
		SensoryItem f = features.get(0);
		assertTrue(f instanceof BallFeature);
		BallFeature bf = (BallFeature)f;
		assertEquals(bf.getDirection(), -1f);
		assertEquals(bf.getDirectionChange(), 9f);
		assertEquals(bf.getDistance(), 98f);
		assertEquals(bf.getDistanceChange(), -14f);
		
		f = features.get(1);
		assertTrue(f instanceof BallFeature);
		bf = (BallFeature)f;
		assertEquals(bf.getDirection(), 0.01f);
		assertEquals(bf.getDirectionChange(), 0f);
		assertEquals(bf.getDistance(), 5f);
		assertEquals(bf.getDistanceChange(), 0f);
		
		f = features.get(2);
		assertTrue(f instanceof BallFeature);
		bf = (BallFeature)f;
		assertEquals(bf.getDirection(), 14f);
		assertEquals(bf.getDirectionChange(), -2f);
		assertEquals(bf.getDistance(), 9f);
		assertEquals(bf.getDistanceChange(), 9.7f);
	}
	
	
	//FIXME The test only covers 66.4% of the code, far more tests are needed urgently
	
}
