/** FlagFeatureTest.java in the package tests.junit.org.RCSImitate.sensoryItems of the RCSImitate project.
    Originally created 15-Nov-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package tests.junit.org.RCSImitate.sensoryItems;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import org.RCSImitate.sensoryItems.FlagFeature;
import org.junit.Test;

/** Tests for the org.RCSImitate.sensoryItems.FlagFeature package
 * 
 * @author Michael W. Floyd
 * @since 0.2
 */
public class FlagFeatureTest {

	/** Tests to make sure the default parameters are set as expected.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void constructor_defaultValues(){
		FlagFeature ff = new FlagFeature();
		assertEquals(ff.getBox(),FlagFeature.c_NO_BOX);
		assertEquals(ff.getFlagNumber(), FlagFeature.c_NO_NUMBER);
		assertEquals(ff.getHorizontalPosition(), FlagFeature.c_UNKNOWN_POSITION);
		assertEquals(ff.getVerticalPosition(), FlagFeature.c_UNKNOWN_POSITION);
		assertFalse(ff.isOutOfBounds());	
	}
	
	/** Gives the function a null parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setHorizontalPosition_null(){
		FlagFeature ff = new FlagFeature();
		ff.setHorizontalPosition(null);
	}
	
	/** Tests setting the horizontal position to
	 * acceptable values.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void setHorizontalPosition_valid(){
		FlagFeature ff = new FlagFeature();
		ff.setHorizontalPosition(FlagFeature.c_CENTER);
		assertEquals(ff.getHorizontalPosition(), FlagFeature.c_CENTER);
		ff.setHorizontalPosition(FlagFeature.c_LEFT);
		assertEquals(ff.getHorizontalPosition(), FlagFeature.c_LEFT);
		ff.setHorizontalPosition(FlagFeature.c_RIGHT);
		assertEquals(ff.getHorizontalPosition(), FlagFeature.c_RIGHT);
	}
	
	/** Tests setting the horizontal position with
	 * an illegal value
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setHorizontalPosition_invalid(){
		FlagFeature ff = new FlagFeature();
		ff.setHorizontalPosition("q");
	}
	
	/** Gives the function a null parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setVerticalPosition_null(){
		FlagFeature ff = new FlagFeature();
		ff.setVerticalPosition(null);
	}
	
	/** Tests setting the vertical position to
	 * acceptable values.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void setVerticalPosition_valid(){
		FlagFeature ff = new FlagFeature();
		ff.setVerticalPosition(FlagFeature.c_CENTER);
		assertEquals(ff.getVerticalPosition(), FlagFeature.c_CENTER);
		ff.setVerticalPosition(FlagFeature.c_TOP);
		assertEquals(ff.getVerticalPosition(), FlagFeature.c_TOP);
		ff.setVerticalPosition(FlagFeature.c_BOTTOM);
		assertEquals(ff.getVerticalPosition(), FlagFeature.c_BOTTOM);
	}
	
	/** Tests setting the vertical position with
	 * an illegal value
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setVerticalPosition_invalid(){
		FlagFeature ff = new FlagFeature();
		ff.setVerticalPosition("q");
	}
	
	/** Gives the function a null parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setBox_null(){
		FlagFeature ff = new FlagFeature();
		ff.setBox(null);
	}
	
	/** Tests setting the box to
	 * acceptable values.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void setBox_valid(){
		FlagFeature ff = new FlagFeature();
		ff.setBox(FlagFeature.c_PENALTY_BOX);
		assertEquals(ff.getBox(),FlagFeature.c_PENALTY_BOX);
		ff.setBox(FlagFeature.c_GOAL_BOX);
		assertEquals(ff.getBox(),FlagFeature.c_GOAL_BOX);
	}
	
	/** Tests setting the box with
	 * an illegal value
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setBox_invalid(){
		FlagFeature ff = new FlagFeature();
		ff.setBox("e");
	}
	
	/** Tests setting if the ball is out of
	 * bounds or not.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void setOutOfBounds(){
		FlagFeature ff = new FlagFeature();
		ff.setOutOfBounds(true);
		assertTrue(ff.isOutOfBounds());
		ff.setOutOfBounds(false);
		assertFalse(ff.isOutOfBounds());
	}
	
	/** Tests setting the flag number
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void setFlagNumber(){
		FlagFeature ff = new FlagFeature();
		for(int ii=0;ii<1000;ii++){
			ff.setFlagNumber(ii);
			assertEquals(ff.getFlagNumber(),ii);
		}
	}
	
	/** Tests equals with a null value
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_null(){
		FlagFeature ff = new FlagFeature();
		assertFalse(ff.equals(null));
	}
	
	
	/** Tests equals with a different class
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_wrongType(){
		FlagFeature ff = new FlagFeature();
		String s = new String();
		assertFalse(ff.equals(s));
	}
	
	/** Tests equals with two default constructed
	 * FlagFeatures
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_default(){
		FlagFeature ff1 = new FlagFeature();
		FlagFeature ff2 = new FlagFeature();
		assertEquals(ff1,ff2);
	}
	
	/** Tests the equality of two flags with same
	 * attributes.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_same(){
		FlagFeature ff1 = new FlagFeature();
		ff1.setBox(FlagFeature.c_PENALTY_BOX);
		ff1.setHorizontalPosition(FlagFeature.c_LEFT);
		ff1.setVerticalPosition(FlagFeature.c_TOP);
		ff1.setOutOfBounds(true);
		ff1.setFlagNumber(40);

		FlagFeature ff2 = new FlagFeature();
		ff2.setBox(FlagFeature.c_PENALTY_BOX);
		ff2.setHorizontalPosition(FlagFeature.c_LEFT);
		ff2.setVerticalPosition(FlagFeature.c_TOP);
		ff2.setOutOfBounds(true);
		ff2.setFlagNumber(40);
		
		assertEquals(ff1,ff2);
	}
	
	/** Tests the equality of two flags that
	 * are on different boxes.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_differentBox(){
		FlagFeature ff1 = new FlagFeature();
		ff1.setBox(FlagFeature.c_PENALTY_BOX);
		FlagFeature ff2 = new FlagFeature();
		ff2.setBox(FlagFeature.c_GOAL_BOX);
		assertFalse(ff1.equals(ff2));
	}
	
	/** Tests the equality of flags with different
	 * horizontal positions.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_differentHorizontal(){
		FlagFeature ff1 = new FlagFeature();
		ff1.setHorizontalPosition(FlagFeature.c_LEFT);
		FlagFeature ff2 = new FlagFeature();
		ff2.setHorizontalPosition(FlagFeature.c_RIGHT);
		FlagFeature ff3 = new FlagFeature();
		ff3.setHorizontalPosition(FlagFeature.c_CENTER);
		
		assertFalse(ff1.equals(ff2));
		assertFalse(ff1.equals(ff3));
		assertFalse(ff3.equals(ff2));
	}
	
	/** Tests the equality of flags with different
	 * vertical positions.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_differentVertical(){
		FlagFeature ff1 = new FlagFeature();
		ff1.setVerticalPosition(FlagFeature.c_TOP);
		FlagFeature ff2 = new FlagFeature();
		ff2.setVerticalPosition(FlagFeature.c_BOTTOM);
		FlagFeature ff3 = new FlagFeature();
		ff3.setVerticalPosition(FlagFeature.c_CENTER);
		
		assertFalse(ff1.equals(ff2));
		assertFalse(ff1.equals(ff3));
		assertFalse(ff3.equals(ff2));
	}
	
	/** Tests the equality of flags with different
	 * out of bounds.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_differentOutOfBounds(){
		FlagFeature ff1 = new FlagFeature();
		ff1.setOutOfBounds(true);
		FlagFeature ff2 = new FlagFeature();
		ff2.setOutOfBounds(false);
		
		assertFalse(ff1.equals(ff2));
	}
	
	/** Tests the equality of flags with different
	 * flag numbers.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_differentNumber(){
		FlagFeature ff1 = new FlagFeature();
		ff1.setFlagNumber(30);
		FlagFeature ff2 = new FlagFeature();
		ff2.setFlagNumber(50);
		
		assertFalse(ff1.equals(ff2));
	}
	
	/** Tests if an object is equal to itself
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_self(){
		FlagFeature ff = new FlagFeature();
		assertTrue(ff.equals(ff));
	}
	
	/** Tests equals if parameters are the same
	 * but the superclass equals condition does not
	 * hold (different distance/direction changes)
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_superDifferent(){
		FlagFeature ff1 = new FlagFeature();
		FlagFeature ff2 = new FlagFeature();
		ff2.setDirectionChange(9.5f);
		ff2.setDistanceChange(-3.2f);
		assertFalse(ff1.equals(ff2));
	}
}
