/** LineFeatureTest.java in the package tests.junit.org.RCSImitate.sensoryItems of the RCSImitate project.
    Originally created 9-Nov-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package tests.junit.org.RCSImitate.sensoryItems;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import junit.framework.JUnit4TestAdapter;

import org.RCSImitate.sensoryItems.LineFeature;
import org.junit.Test;

/** Tests the org.RCSImitate.sensoryItems.LineFeature class
 * 
 * @author Michael W. Floyd
 * @since 0.2
 */
public class LineFeatureTest {

	/** Ensure the feature name is set properly.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void constructor_checkName(){
		LineFeature lf = new LineFeature();
		assertEquals(lf.getFeatureName(),LineFeature.c_LINE);
	}
	
	/** Ensure the default constructor set the location
	 * to unknown
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void constructor_defaultLocation(){
		LineFeature lf = new LineFeature();
		assertEquals(lf.getLocation(),LineFeature.c_LOCATION_UNKNOWN);
	}
	
	/** Test to see if a location of 'l' (left)
	 * is accepted
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void setLocation_l(){
		LineFeature lf = new LineFeature();
		lf.setLocation(LineFeature.c_LOCATION_LEFT);
		assertEquals(lf.getLocation(),LineFeature.c_LOCATION_LEFT);
	}
	
	/** Test to see if a location of 'r' (right)
	 * is accepted
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void setLocation_r(){
		LineFeature lf = new LineFeature();
		lf.setLocation(LineFeature.c_LOCATION_RIGHT);
		assertEquals(lf.getLocation(),LineFeature.c_LOCATION_RIGHT);
	}
	
	/** Test to see if a location of 't' (top)
	 * is accepted
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void setLocation_t(){
		LineFeature lf = new LineFeature();
		lf.setLocation(LineFeature.c_LOCATION_TOP);
		assertEquals(lf.getLocation(),LineFeature.c_LOCATION_TOP);
	}
	
	/** Test to see if a location of 'b' (bottom)
	 * is accepted
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void setLocation_b(){
		LineFeature lf = new LineFeature();
		lf.setLocation(LineFeature.c_LOCATION_BOTTOM);
		assertEquals(lf.getLocation(),LineFeature.c_LOCATION_BOTTOM);
	}
	
	/** Test with an invalid location
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setLocation_invalid(){
		LineFeature lf = new LineFeature();
		lf.setLocation("q");
	}
	
	/** Try setting the location to null
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IllegalArgumentException.class)
	public void setLocation_null(){
		LineFeature lf = new LineFeature();
		lf.setLocation(null);
	}
	
	/** Test when the second object is null
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_null(){
		LineFeature lf = new LineFeature();
		assertFalse(lf.equals(null));
	}
	
	/** Test when one object is of differnt type
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_wrongType(){
		LineFeature lf = new LineFeature();
		String s = new String();
		assertFalse(lf.equals(s));
	}
	
	/** Test two objects made with default constructor
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_default(){
		LineFeature lf1 = new LineFeature();
		LineFeature lf2 = new LineFeature();

		assertEquals(lf1,lf2);
	}
	
	/** Make sure two objects with the same location are equal
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_sameLocation(){
		LineFeature lf1 = new LineFeature();
		lf1.setLocation(LineFeature.c_LOCATION_BOTTOM);
		LineFeature lf2 = new LineFeature();
		lf2.setLocation(LineFeature.c_LOCATION_BOTTOM);
		
		assertEquals(lf1,lf2);
		
		lf1 = new LineFeature();
		lf1.setLocation(LineFeature.c_LOCATION_TOP);
		lf2 = new LineFeature();
		lf2.setLocation(LineFeature.c_LOCATION_TOP);
		
		assertEquals(lf1,lf2);
		
		lf1 = new LineFeature();
		lf1.setLocation(LineFeature.c_LOCATION_LEFT);
		lf2 = new LineFeature();
		lf2.setLocation(LineFeature.c_LOCATION_LEFT);
		
		assertEquals(lf1,lf2);
		
		lf1 = new LineFeature();
		lf1.setLocation(LineFeature.c_LOCATION_RIGHT);
		lf2 = new LineFeature();
		lf2.setLocation(LineFeature.c_LOCATION_RIGHT);
		
		assertEquals(lf1,lf2);
	}
	
	/** Tests the equality of two objects with different locations.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_differntLocation(){
		LineFeature lf1 = new LineFeature();
		lf1.setLocation(LineFeature.c_LOCATION_BOTTOM);
		LineFeature lf2 = new LineFeature();
		lf2.setLocation(LineFeature.c_LOCATION_RIGHT);
		
		assertFalse(lf1.equals(lf2));
		
		lf2 = new LineFeature();
		lf2.setLocation(LineFeature.c_LOCATION_LEFT);
		
		assertFalse(lf1.equals(lf2));
		
		lf2 = new LineFeature();
		lf2.setLocation(LineFeature.c_LOCATION_TOP);
		
		assertFalse(lf1.equals(lf2));
		
	}
	
	/** Tests equality to itself
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_self(){
		LineFeature lf = new LineFeature();
		assertTrue(lf.equals(lf));
	}
	
	/** Tests the equality when the LineFeature variables
	 * are the same, but the superclass variables are different
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_superDifferent(){
		LineFeature lf1 = new LineFeature();
		LineFeature lf2 = new LineFeature();
		lf2.setDirectionChange(-0.3f);
		lf2.setDistanceChange(44.2f);
		assertFalse(lf1.equals(lf2));
	}
	
	public static junit.framework.Test suite(){
		return new JUnit4TestAdapter(LineFeatureTest.class);
	}
	
}
