/** PlayerFeatureTest.java in the package tests.junit.org.RCSImitate.sensoryItems of the RCSImitate project.
    Originally created 10-Nov-07

    Copyright (C) 2007  Michael W. Floyd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 * 
 */

package tests.junit.org.RCSImitate.sensoryItems;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import org.RCSImitate.sensoryItems.PlayerFeature;
import org.junit.Test;

import junit.framework.JUnit4TestAdapter;

/** Tests for the org.RCSImitiate.features.PlayerFeatureTest class
 * 
 * @author Michael W. Floyd
 * @since 0.2
 */
public class PlayerFeatureTest {

	/** Test the constructor with a null
	 * parameter.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test(expected=IllegalArgumentException.class)
	public void constructor_null(){
		new PlayerFeature(null);
	}
	
	/** Tests to see the feature name is properly
	 * set by the constructor.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void constructor_name(){
		PlayerFeature pf = new PlayerFeature("myFeature");
		assertEquals(pf.getFeatureName(),"myFeature");
	}
	
	/** Tests to see the is goalie is properly
	 * set by the constructor.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void constructor_goalie(){
		PlayerFeature pf = new PlayerFeature("myFeature");
		assertEquals(pf.getBodyDirection(),0.0f);
	}
	
	/** Tests to see the uniform is properly
	 * set by the constructor.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void constructor_uniform(){
		PlayerFeature pf = new PlayerFeature("myFeature");
		assertEquals(pf.getHeadDirection(),0.0f);
	}
	
	/** Tests to see the head direction is properly
	 * set by the constructor.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void constructor_headDirection(){
		PlayerFeature pf = new PlayerFeature("myFeature");
		assertEquals(pf.isGoalie(),false);
	}
	
	/** Tests to see the body direction is properly
	 * set by the constructor.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void constructor_bodyDirection(){
		PlayerFeature pf = new PlayerFeature("myFeature");
		assertEquals(pf.getUniformNumber(),PlayerFeature.c_UNIFORM_DEFAULT);
	}
	
	/** Tests if the is goalie can be properly set.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void setIsGoalie(){
		PlayerFeature pf = new PlayerFeature("myFeature");
		pf.setIsGoalie(true);
		assertTrue(pf.isGoalie());
	}
	
	/** Tests if the uniform can be properly set.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void setUniform(){
		PlayerFeature pf = new PlayerFeature("myFeature");
		pf.setUniformNumber(13);
		assertEquals(pf.getUniformNumber(),13);
	}
	
	/** Tests if the head direction can be properly set.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void setHeadDirection(){
		PlayerFeature pf = new PlayerFeature("myFeature");
		pf.setHeadDirection(-2.3f);
		assertEquals(pf.getHeadDirection(),-2.3f);
	}
	
	/** Tests if the body direction can be properly set.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void setBodyDirection(){
		PlayerFeature pf = new PlayerFeature("myFeature");
		pf.setBodyDirection(55.5f);
		assertEquals(pf.getBodyDirection(),55.5f);
	}
	
	/** Tests equals using a null paramater
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_null(){
		PlayerFeature pf = new PlayerFeature("myFeature");
		assertFalse(pf.equals(null));
	}
	
	/** Tests equals using a different object type
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_differentClass(){
		PlayerFeature pf = new PlayerFeature("myFeature");
		String s = new String();
		assertFalse(pf.equals(s));
	}
	
	/** Tests the equality of two objects made with the default
	 * constructor.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_default(){
		PlayerFeature pf1 = new PlayerFeature("myFeature");
		PlayerFeature pf2 = new PlayerFeature("myFeature");
		assertEquals(pf1,pf2);
	}
	
	/** Tests equals with two same objects with parameters
	 * changed from defaults.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_notdefault(){
		PlayerFeature pf1 = new PlayerFeature("myFeature");
		pf1.setIsGoalie(true);
		pf1.setUniformNumber(3);
		pf1.setBodyDirection(-0.34f);
		pf1.setHeadDirection(45.0f);
		
		PlayerFeature pf2 = new PlayerFeature("myFeature");
		pf2.setIsGoalie(true);
		pf2.setUniformNumber(3);
		pf2.setBodyDirection(-0.34f);
		pf2.setHeadDirection(45.0f);
		
		assertEquals(pf1,pf2);
	}
	
	/** Tests equals when they have different
	 * feature names.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_differentName(){
		PlayerFeature pf1 = new PlayerFeature("myFeature");
		pf1.setIsGoalie(true);
		pf1.setUniformNumber(3);
		pf1.setBodyDirection(-0.34f);
		pf1.setHeadDirection(45.0f);
		
		PlayerFeature pf2 = new PlayerFeature("player");
		pf2.setIsGoalie(true);
		pf2.setUniformNumber(3);
		pf2.setBodyDirection(-0.34f);
		pf2.setHeadDirection(45.0f);
		
		assertFalse(pf1.equals(pf2));
	}
	
	/** Tests equals when they have different
	 * is goalie.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_differentGoalie(){
		PlayerFeature pf1 = new PlayerFeature("myFeature");
		pf1.setIsGoalie(true);
		pf1.setUniformNumber(3);
		pf1.setBodyDirection(-0.34f);
		pf1.setHeadDirection(45.0f);
		
		PlayerFeature pf2 = new PlayerFeature("myFeature");
		pf2.setIsGoalie(false);
		pf2.setUniformNumber(3);
		pf2.setBodyDirection(-0.34f);
		pf2.setHeadDirection(45.0f);
		
		assertFalse(pf1.equals(pf2));
	}
	
	/** Tests equals when they have different
	 * uniform.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_differentUniform(){
		PlayerFeature pf1 = new PlayerFeature("myFeature");
		pf1.setIsGoalie(true);
		pf1.setUniformNumber(3);
		pf1.setBodyDirection(-0.34f);
		pf1.setHeadDirection(45.0f);
		
		PlayerFeature pf2 = new PlayerFeature("myFeature");
		pf2.setIsGoalie(true);
		pf2.setUniformNumber(8);
		pf2.setBodyDirection(-0.34f);
		pf2.setHeadDirection(45.0f);
		
		assertFalse(pf1.equals(pf2));
	}
	
	/** Tests equals when they have different
	 * head direction.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_differentHeadDirection(){
		PlayerFeature pf1 = new PlayerFeature("myFeature");
		pf1.setIsGoalie(true);
		pf1.setUniformNumber(3);
		pf1.setBodyDirection(-0.34f);
		pf1.setHeadDirection(19.3f);
		
		PlayerFeature pf2 = new PlayerFeature("myFeature");
		pf2.setIsGoalie(true);
		pf2.setUniformNumber(3);
		pf2.setBodyDirection(-0.34f);
		pf2.setHeadDirection(45.0f);
		
		assertFalse(pf1.equals(pf2));
	}
	
	/** Tests equals when they have different
	 * body direction.
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_differentBodyDirection(){
		PlayerFeature pf1 = new PlayerFeature("myFeature");
		pf1.setIsGoalie(true);
		pf1.setUniformNumber(3);
		pf1.setBodyDirection(-0.34f);
		pf1.setHeadDirection(45.0f);
		
		PlayerFeature pf2 = new PlayerFeature("myFeature");
		pf2.setIsGoalie(true);
		pf2.setUniformNumber(3);
		pf2.setBodyDirection(5.4f);
		pf2.setHeadDirection(45.0f);
		
		assertFalse(pf1.equals(pf2));
	}
	
	/** Tests to make sure a PlayerFeature is equal to itself
	 * 
	 * @author Michael W. Floyd
	 * @since 0.2
	 */
	@Test
	public void equals_self(){
		PlayerFeature pf = new PlayerFeature("someFeat");
		assertTrue(pf.equals(pf));
	}
	
	public static junit.framework.Test suite(){
		return new JUnit4TestAdapter(PlayerFeatureTest.class);
	}
}
