/*
 * Created on Oct 31, 2004
 *
 * This is a simple implementation of a cartesian distance calculation involving all the objects in the scene
 * (lines, flags, players, goal, ball)
 * 
 * this version sorts the objects into individual lists of ball, goal, players, etc. and then calculates 
 * distances on like objects: ball with other ball, players with other players, etc.
 * 
 * changelog
 *   january 2005: updated to include support for variable weights
 */
package sceneMatch.nearest_neighbor_matching;

import java.util.*;

import sceneInfo.ObjectInfo;
import sceneInfo.Scene;

/**
 * @author Kevin Lam
 *
 */
public class NearestNeighborCartesianObjects extends NearestNeighborParent
{
	private float penalty = 100;
	
	char friendlyside ='l';
	char enemyside = 'r';
		
	/**
	 * @param k
	 */
	public NearestNeighborCartesianObjects(int k, char side)
	{
		super(k, side);
		if (side == 'r') {
			friendlyside = 'r';
			enemyside= 'l';
		}

	}

	/* (non-Javadoc)
	 * @see rcscene.DistanceCalculation#distanceBetween(rcscene.VisionTable, rcscene.VisionTable)
	 */
	public float distanceBetween(Scene one, Scene two)
	{
		float totalDist = 0;
		// get lists of each type of object, and run distance calculation for each
		// then take sum of all distances between sets of objects
		if (weightBalls > 0)
		{
			totalDist += getDistanceForObjectList(one.getBallObjects().toArray(), 
											  two.getBallObjects().toArray()) * weightBalls;
		}
		
		// choice between locating any/all players or break into opponents/teammates/unknowns
		if (weightAllPlayers > 0)
		{
			totalDist += getDistanceForObjectList(one.getPlayerObjects().toArray(), 
											  two.getPlayerObjects().toArray()) * weightAllPlayers;
		}

		if (weightTeammates > 0)
		{
			totalDist += getDistanceForObjectList(one.getTeammateObjects().toArray(), 
											  two.getTeammateObjects().toArray()) * weightTeammates;
		}
		
		if (weightOpponents > 0)
		{
			totalDist += getDistanceForObjectList(one.getOpponentObjects().toArray(), 
											  two.getOpponentObjects().toArray()) * weightOpponents;
		}

		if (weightUnknownPlayers > 0)
		{		
			totalDist += getDistanceForObjectList(one.getUnknownPlayerObjects().toArray(), 
											  two.getUnknownPlayerObjects().toArray()) * weightUnknownPlayers;
		}
			
		// should have opponent/team goal differentiation
		if (weightGoals > 0)
		{
			totalDist += getDistanceForObjectList(one.getGoalObjects().toArray(), 
											  two.getGoalObjects().toArray()) * weightGoals;
		}
		
		// not really valid unless we can logically correspond by line types (but we can't get too specific, either...)
		if (weightLines > 0)
		{
			totalDist += getDistanceForObjectList(one.getLineObjects().toArray(), 
											  two.getLineObjects().toArray()) * weightLines;
		}
		
		// not really valid unless we can logically correspond by flag types
		if (weightFlags > 0)
		{
			totalDist += getDistanceForObjectList(one.getFlagObjects().toArray(), 
											  two.getFlagObjects().toArray()) * weightFlags;
		}
		
		return totalDist;
	}
	
	// simple heuristic for matching objects: sort them by distance from closest to farthest, then pair off
	public float getDistanceForObjectList(Object[] objectsOne, Object[] objectsTwo)
	{
		int sceneOneSize = objectsOne.length;
		sortObjectsByDistance(objectsOne);
		
		int sceneTwoSize = objectsTwo.length;
		sortObjectsByDistance(objectsTwo);
		
		int smaller = (sceneOneSize < sceneTwoSize ? sceneOneSize :sceneTwoSize );
		int larger = (sceneOneSize > sceneTwoSize ? sceneOneSize :sceneTwoSize );
		
		float totalDist = 0;
		
		for (int i = 0; i < smaller; i++)
		{
			float r1 = ((ObjectInfo) objectsOne[i]).getDistance();
			float th1 = ((ObjectInfo) objectsOne[i]).getDirection();
			float r2 = ((ObjectInfo) objectsTwo[i]).getDistance();
			float th2 = ((ObjectInfo) objectsTwo[i]).getDirection();

			totalDist += polarDistance(r1, th1, r2, th2);
			
		}
		
		// assess penalty for all objects in one list but not in another?
		// realistically should assess penalty based on some other factors, such as distance
		totalDist += (larger - smaller) * penalty;
		
		return totalDist;
	}

	// sorts the array by the distance attribute in ObjectInfo
	public void sortObjectsByDistance(Object [] sceneObjects)
	{
		Arrays.sort(sceneObjects, new Comparator()
			{
				 public int compare(Object o1, Object o2)
				 {
					 float dist1 = ((ObjectInfo) o1).getDistance();
					 float dist2 = ((ObjectInfo) o2).getDistance();
  					 
					 if (dist1 == dist2)
						return 0;
					 else
						return (dist1 > dist2 ? 1 : -1);
				 }
			});
		
	}

}
