//
//	File:			Brain.java
//	Author:		Krzysztof Langner
//	Date:			1997/04/28
//
//    Modified by:	Paul Marlow
//    Modified by:  Kevin Lam (incorporation of scene recognition)
//    modifications pulled in from Group 3 for parameterizable algorithm selection

package visiontable;

import java.io.*;
import java.util.*;


class Brain extends Thread implements SensorInput
{
	
	Vector storedScenes = null;
	int numRows = 0;
	int numCols = 0;
	
	//---------------------------------------------------------------------------
	// This constructor:
	// - stores connection to krislet
	// - starts thread for this object
	public Brain(SendCommand krislet, String team, char side, int number, String playMode, String scenelib, int sceneSel, 
					int distCal, int numBest, float[] objWeights, float[] actionWeights)
	{
		m_timeOver = false;
		m_krislet = krislet;
		m_memory = new Memory();
		m_team = team;
		m_side = side;
//		m_number = number;
//		m_playMode = playMode;

		m_distCal = distCal;
		m_sceneSel = sceneSel;
		m_numBest = numBest;
		m_objWeights = objWeights;
		m_actionWeights = actionWeights;

		// first put it somewhere on my side
		m_krislet.move( -Math.random()*52.5 , Math.random()*34.0 );

		// initialize scene database
		//System.out.println("Loading scene file " + scenelib);
   		loadSceneFile(scenelib);
   		
		start();
	}

	// for a future version maybe try to encapsulate all these into a single utility class that other clients just hook into...
	// ... an item for refactoring later
	public void loadSceneFile(String fname)
		{
			System.out.println("Loading scenes from file " + fname);
			try
			{
				FileInputStream fileIn = new FileInputStream(fname);
				
				ObjectInputStream ois = new ObjectInputStream(fileIn);
				numRows = ((Integer) ois.readObject()).intValue();
				numCols = ((Integer) ois.readObject()).intValue();
				storedScenes = (Vector) ois.readObject();
				  
				ois.close();
				fileIn.close();
			}
			catch (FileNotFoundException e)
			{
				e.printStackTrace();
			}
			catch (IOException e)
			{
				e.printStackTrace();
			}
			catch (ClassNotFoundException e)
			{
				e.printStackTrace();
			}
		
			System.out.println("Read " + storedScenes.size() + " scenes, table size (" + numRows + "," + numCols + ")");
		
		}
		
	//---------------------------------------------------------------------------
	// This is main brain function used to make decision
	// In each cycle we decide which command to issue based on
	// current situation.
	
	//  1. Get current scene info from memory
	//  2. Create a representative Scene and VisionTable
	//  3. Instantiate a DistanceCalculation object
	//  4. Pick out the best scene from the list
	//  5. Implement the same action that was taken from the stored scene
	//
	//	To ensure that we don't send commands to often after each cycle
	//	we waits one simulator steps. (This of course should be done better)
	public void run()
	{
		ObjectInfo object;
		
		DistanceCalculation distanceCalc;			// plug-and-play distance calculation object
				
		// *********************************************
		// choose scene selection algorithm, parameters
		// *********************************************
		switch(m_distCal) {
		  case 0: {
			  distanceCalc = new NearestNeighborCartesianObjects(m_numBest, m_side);
			  break;
			}
		  case 1: {
		    distanceCalc = new NearestNeighborCartesianCellObjects(m_numBest, m_side);
		    break;
		  }
		  case 2: {
		    distanceCalc = new NearestNeighborCellBallGoalDistance(m_numBest, m_side);
		    break;
		  }
		  case 3: {
			distanceCalc = new RandomDistance(m_numBest, m_side);
			break;
		  }
		  default: {
		    distanceCalc = new NearestNeighborCartesianCellObjects(m_numBest, m_side);
		    break;
		  }
		}
		// load weights
		distanceCalc.setObjectWeights(m_objWeights);

		while( !m_timeOver )
		{
			// let's do a test first...
			VisualInfo stuff = m_memory.getInfo();
			if (stuff == null)
			{
				m_memory.waitForNewInfo();
			}
			else {

			// we now have a viable VisualInfo object: now apply transformation from LogToScenes to cell-discretize it
			LogToScenes.parseSeeInfo(stuff, numRows, numCols);
			
			Scene thisScene = new Scene(numRows, numCols);
			thisScene.setVision(stuff);
			
			Scene[] bestScenes = distanceCalc.findClosest(thisScene, storedScenes);
			
			// record start time
			//startTime = System.currentTimeMillis();
			
			
			// result is a set of candidate scenes stored in bestScenes; now, choose the best one to imitate
			
			// *********************************************
			// choose scene selection algorithm, parameters
			// *********************************************
			Scene matched;
			
			switch(m_sceneSel) {
			  case 1: {
			    matched = SceneSelection.ChooseRandom(bestScenes, m_numBest);
			    break;
			  }
			  case 2: {
			    matched = SceneSelection.VoteWeighted(bestScenes, bestScenes.length, m_actionWeights);
			    break;
			  }
			  case 3: {
				matched = SceneSelection.VoteWeightedWithRandom(bestScenes, bestScenes.length, m_actionWeights, 20);
				break;
			  }
			  default: {	// 0
			    matched = SceneSelection.ChooseFirstValid(bestScenes);
			    break;
			  }
			}
			
			// record start time
			//endTime = System.currentTimeMillis();
			//System.out.println("Time taken: " + (endTime - startTime));
			//totalTime += (endTime - startTime);

			// now use the same action as the one we just matched with
			System.out.println("Best match in scene " + matched.getIdentString() + "; actions are " + matched.getActions());
			
			List actions = matched.getActions();
			for (int x = 0; x < actions.size(); x++)
			{
				Action doThis = (Action) actions.get(x);
								
				if (doThis.getAction() == Action.ACTION_DASH)
				{
					m_krislet.dash(doThis.getActionPower() );
					System.out.println("Sending dash " + doThis.getActionPower());
				}
				else if (doThis.getAction() == Action.ACTION_TURN)
				{
					double turnDir = doThis.getActionDirection();
					
					/* are there any exact target matches?
					Vector possibleTargets = LogToScenes.predictTargets(thisScene, doThis, 0);
					if (possibleTargets.size() > 0)
					{
						// we're pretty sure about this
						ObjectInfo obj = (ObjectInfo) possibleTargets.elementAt(0);
						System.out.println("I think I'm supposed to aim at the " + obj.getType());
						// now, find the one WE'VE got, and aim at it instead
						//turnDir = obj.getDirection();
					}*/

					m_krislet.turn(turnDir);
					System.out.println("Sending turn " + turnDir);
				}
				else if (doThis.getAction() == Action.ACTION_TURNNECK) {
					m_krislet.turn_neck(doThis.getActionDirection());
					System.out.println("Sending turn_neck " + doThis.getActionDirection());
				}
				else if (doThis.getAction() == Action.ACTION_KICK)
				{
					double kickDir = doThis.getActionDirection();
					 
					/* are there any exact target matches?
					Vector possibleTargets = LogToScenes.predictTargets(thisScene, doThis, 0);
					if (possibleTargets.size() > 0)
					{
						// we're pretty sure about this
						ObjectInfo obj = (ObjectInfo) possibleTargets.elementAt(0);
						System.out.println("I think I'm supposed to aim at the " + obj.getType());
						// now, find the one WE'VE got, and aim at it instead
					}
					*/

					m_krislet.kick(doThis.getActionPower(), kickDir);
					System.out.println("Sending kick " + doThis.getActionPower() + ", " + kickDir);
				}
				// catch??

			}
			}

			// sleep one step to ensure that we will not send
			// two commands in one cycle.
			try{
				Thread.sleep(2*SoccerParams.simulator_step);
			}catch(Exception e){}
		}
		return;
	}



//===========================================================================
// Here are suporting functions for implement logic


//===========================================================================
// Implementation of SensorInput Interface


	//---------------------------------------------------------------------------
	// This function sends see information
	public void see(VisualInfo info)
	{
		m_memory.store(info);
	}


	//---------------------------------------------------------------------------
	// This function receives hear information from player
	public void hear(int time, int direction, String message)
	{
		//hear(time, message);
	}

	//---------------------------------------------------------------------------
	// This function receives hear information from referee
	public void hear(int time, String message)
	{	
		try
		{					 
		StringTokenizer tokenizer = new StringTokenizer(message,"() ");
		String token;

		// First is referee token and time token
		tokenizer.nextToken();	//hear
		tokenizer.nextToken();  //time
		tokenizer.nextToken();  //referee
		token = tokenizer.nextToken();

		if(token.compareTo("time_over") == 0)
		{
			m_timeOver = true;
		}
		
		}
		catch (Exception e)
		{
		}
			
	}


//===========================================================================
// Private members
	private SendCommand				m_krislet;			// robot which is controled by this brain
	private Memory					m_memory;				// place where all information is stored
	private char					m_side;
	private String					m_team;
	private	int						m_distCal;
	private int						m_sceneSel;
	private int						m_numBest;
	private float[]					m_objWeights;
	private float[]					m_actionWeights;
	
	volatile private boolean		m_timeOver;
}