/*
 * 
 * Created on Mar 1, 2004
 *
 * 03/29/2004: initial version
 * 10/19/2004: refactored to new version which directly references VisualInfos
 * 11/15/2004: included team-name change from Deryck Velasquez
 * 01/23/2005: added methods to access teammate/opponent lists
 * 
 * 
 */
package visiontable;

import java.io.Serializable;
import java.util.*;

/**
 * @author Kevin Lam
 *
 * A Scene is a representation of both a set of visuals (a VisualInfo) and a set of actions
 * associated with those visuals.
 * 
 */
public class Scene implements Serializable
{

	private VisualInfo vision = null;
	private Vector actions = null;
	private String identString = null;		// typically a time index, but open to anything
	private int tableRows = 1;				// default sizes for cells
	private int tableColumns = 1;
    private String teamName = null;			// for RoboCup team names
	
	/**
	 * 
	 */
	public Scene()
	{
		super();
		actions = new Vector();
	}

	/**
	 * 
	 */
	public Scene(int numRows, int numCols)
	{
		super();
		tableRows = numRows;
		tableColumns = numCols;
		actions = new Vector();
	}

	/**
	 * 
	 */
	public Scene(VisualInfo vis, Action act)
	{
		super();
		actions = new Vector();
		vision = vis;
		actions.add(act);
	}

	/**
	 * @return
	 */
	public Vector getActions()
	{
		return actions;
	}

	/**
	 * @param action
	 */
	public void addAction(Action action)
	{
		actions.add( action );
	}

	/**
	 * @param action
	 */
	public void removeAction(Action action)
	{
		actions.remove( action );
	}

	// may need to redo the "actions" equals test
	// testing for exact matches, as opposed to identical sets
	public boolean equals(Scene other)
	{
		if ( (actions.equals(other.getActions())) && (vision.equals(other.getVision())) )
		{
			return true;
		}
		else
		{
			return false;
		}
	}
	/**
	 * Don't get the VisualInfo manually; instead, use the other getter methods to get by type or by cell location
	 * @return
	 */
	public VisualInfo getVision()
	{
		return vision;
	}

	public Vector getAllObjects()
	{
		if (vision == null)
			return new Vector();
		else
			return vision.m_objects;
	}
	
	public Vector getObjectsAt(int col, int row)
	{
		Vector results = new Vector();
		if (vision != null)
		{
			// iterate through all objects; copy the ones whose location matches this cell
			Iterator it = vision.m_objects.iterator();
			while (it.hasNext())
			{
				ObjectInfo obj = (ObjectInfo) it.next();
				if ( (obj.m_tableCol == col) && (obj.m_tableRow == row) )
				{
					// match
					results.add(obj);
				}
			}
		}

		return results;
	}
	
	public Vector getBallObjects()
	{
		if (vision == null)
			return new Vector();
		else
			return vision.getBallList();
	}

	public Vector getLineObjects()
	{
		if (vision == null)
			return new Vector();
		else
			return vision.getLineList();
	}
		
	public Vector getFlagObjects()
	{
		if (vision == null)
			return new Vector();
		else
			return vision.getFlagList();
	}
	
	public Vector getPlayerObjects()
	{
		if (vision == null)
			return new Vector();
		else
			return vision.getPlayerList();
	}
	public Vector getTeammateObjects()
	{
		if (vision == null)
			return new Vector();
		else
			return vision.getTeammatesList();
	}
	public Vector getOpponentObjects()
	{
		if (vision == null)
			return new Vector();
		else
			return vision.getOpponentsList();
	}
	public Vector getUnknownPlayerObjects()
	{
		if (vision == null)
			return new Vector();
		else
			return vision.getUnknownPlayersList();
	}
	
	public Vector getGoalObjects()
	{
		if (vision == null)
			return new Vector();
		else
			return vision.getGoalList();
	}

	/**
	 * @param vis
	 */
	public void setVision(VisualInfo vis)
	{
		vision = vis;
	}

	/**
	 * @return
	 */
	public String getIdentString()
	{
		return identString;
	}

	/**
	 * @param string
	 */
	public void setIdentString(String string)
	{
		identString = string;
	}

	/**
	 * @author Deryck Velasquez
         * @version 1.0
	 */
	public String getTeamName()
	{
		return teamName;
	}

	/**
	 * @author Deryck Velasquez
         * @version 1.0
	 */
	public void setTeamName(String _teamName)
	{
		teamName = _teamName;
	}        
        
	// other Scene functions here (formerly defined in VisionTable)

	/**
	 * @param string
	 */
	public String outputTable()
	{
		if (vision == null)
			return "no table";
		else
		{
			String x = new String("");
			x = x.concat(outputRowMarkings());
			for (int i = 0; i < tableRows; i++)
			{
				for(int j = 0; j < tableColumns; j++)
				{
					x = x.concat("|" + outputCell(tableRows - i - 1, j) );
				}
				x = x.concat("|\n");
				x = x.concat(outputRowMarkings());
			}
				
			return x;
		}
	}
	
	private String outputCell(int row, int column)
	{
		String x = new String();
		Vector stuff = getObjectsAt(column, row);

		// output format is:  "BG 3P3F3L"
		x = x.concat( getBallCount(stuff) > 0 ? "B" : " ");
		x = x.concat( getGoalCount(stuff) > 0 ? "G " : "  ");

		int count = getPlayerCount(stuff);
		x = x.concat( count > 0 ? count + "P" : "  ");
		count = getFlagCount(stuff);
		x = x.concat( count > 0 ? count + "F" : "  ");
		count = getLineCount(stuff);
		x = x.concat( count > 0 ? count + "L" : "  ");
		
		return x;
	}
	private String outputRowMarkings()
	{
		String x = new String();
		for(int i = 0; i < tableColumns; i++)
		{
			x = x.concat("+---------");
		}
		x = x.concat("+\n");
		return x;
	}

	/**
	 * @return
	 */
	public int getTableColumns()
	{
		return tableColumns;
	}

	/**
	 * @return
	 */
	public int getTableRows()
	{
		return tableRows;
	}

	/**
	 * @param i
	 */
	public void setTableColumns(int i)
	{
		tableColumns = i;
	}

	/**
	 * @param i
	 */
	public void setTableRows(int i)
	{
		tableRows = i;
	}

	public int getPlayerCount(Vector list)
	{
			int c = 0;
        
			for (int i = 0; i < list.size(); i++)
			{
					ObjectInfo x = (ObjectInfo) list.elementAt(i);
					if (x.m_type.startsWith("player"))  
							c++;
			}
			return c;
	}

	public int getBallCount(Vector list)
	{
			int c = 0;
        
			for (int i = 0; i < list.size(); i++)
			{
					ObjectInfo x = (ObjectInfo) list.elementAt(i);
					if (x.m_type.startsWith("ball"))  
							c++;
			}
			return c;
	}

	public int getGoalCount(Vector list)
	{
			int c = 0;
        
			for (int i = 0; i < list.size(); i++)
			{
					ObjectInfo x = (ObjectInfo) list.elementAt(i);
					if (x.m_type.startsWith("goal"))  
							c++;
			}
			return c;
	}

	public int getFlagCount(Vector list)
	{
			int c = 0;
        
			for (int i = 0; i < list.size(); i++)
			{
					ObjectInfo x = (ObjectInfo) list.elementAt(i);
					if (x.m_type.startsWith("flag"))  
							c++;
			}
			return c;
	}

	public int getLineCount(Vector list)
	{
			int c = 0;
        
			for (int i = 0; i < list.size(); i++)
			{
					ObjectInfo x = (ObjectInfo) list.elementAt(i);
					if (x.m_type.startsWith("line"))  
							c++;
			}
			return c;
	}

}
