/*
 * This class is based on Krislet's Brain and provides offline training/validation services to the KrisletScenes client.
 * It can be used as a testbed for statistically evaluating the performance of DistanceCalculation objects.
 * 
 * Created on Jan 26, 2005
 *
 */
package visiontable;

import java.util.*;
import java.io.*;

/**
 * @author Kevin Lam
 *
 */
class Validator
{
	
	private Vector storedScenes = new Vector();
	private Vector validateScenes = new Vector();
	
	int numRows = 0;
	int numCols = 0;

	public Validator(String scenelib, String comparelib, int sceneSel, 
					int distCal, int numBest, float[] objWeights, float[] actionWeights)
	{
		m_side = 'l';	// arbitrary, shouldn't matter

		m_distCal = distCal;
		m_sceneSel = sceneSel;
		m_numBest = numBest;
		m_objWeights = objWeights;
		m_actionWeights = actionWeights;

		// initialize scene database
		System.out.println("Loading stored scene file " + scenelib);
		loadSceneFile(scenelib, storedScenes);
		
		int srows = numRows;
		int scols = numCols;
		
		if(comparelib.compareToIgnoreCase(scenelib) == 0)
		{
			System.out.println("Using the same scene file for target");
			validateScenes = storedScenes;
		}
		else
		{
			System.out.println("Loading target scene file " + comparelib);
			loadSceneFile(comparelib, validateScenes);
		} 
   		
   		// double check scene files match in dimension
		if((srows != numRows)||(scols != numCols))
		{
			System.err.println("Can't run validation - scene files differ in dimension");
		}
	}

	// for a future version maybe try to encapsulate all these into a single utility class that other clients just hook into...
	// ... an item for refactoring later
	public void loadSceneFile(String fname, Vector target)
		{
			System.out.println("Loading scenes from file " + fname);
			try
			{
				FileInputStream fileIn = new FileInputStream(fname);
				
				ObjectInputStream ois = new ObjectInputStream(fileIn);
				numRows = ((Integer) ois.readObject()).intValue();
				numCols = ((Integer) ois.readObject()).intValue();
				target.addAll((Vector) ois.readObject());
				  
				ois.close();
				fileIn.close();
			}
			catch (FileNotFoundException e)
			{
				e.printStackTrace();
			}
			catch (IOException e)
			{
				e.printStackTrace();
			}
			catch (ClassNotFoundException e)
			{
				e.printStackTrace();
			}
		
			System.out.println("Read " + target.size() + " scenes, table size (" + numRows + "," + numCols + ")");
		
		}
		
	//  1. Get next Scene from target list
	//  3. Instantiate a DistanceCalculation object
	//  4. Pick out the best scene from the list
	//  5. Compare chosen action with stored action and get statistics
	//
	public void runValidation()
	{
		long startTime = 0, endTime = 0, totalTime = 0;
		float totalMatches = 0, totalCompares = 0;
		float origKick = 0, correctKick = 0;
		int totalScenes = 0;
		
		System.out.println("Beginning validation of " + validateScenes.size() + " scenes");
		
		DistanceCalculation distanceCalc;			// plug-and-play distance calculation object
				
		// *********************************************
		// choose scene selection algorithm, parameters
		// *********************************************
		switch(m_distCal) {
		  case 0: {
			  distanceCalc = new NearestNeighborCartesianObjects(m_numBest, m_side);
			  //distanceCalc = new NearestNeighborBallGoalDistance(m_numBest, m_side);
			  break;
			}
		  case 1: {
			distanceCalc = new NearestNeighborCartesianCellObjects(m_numBest, m_side);
			break;
		  }
		  case 2: {
			distanceCalc = new NearestNeighborCellBallGoalDistance(m_numBest, m_side);
			break;
		  }
		  case 3: {
			distanceCalc = new RandomDistance(m_numBest, m_side);
			break;
		  }
		  default: {
			distanceCalc = new NearestNeighborCartesianCellObjects(m_numBest, m_side);
			break;
		  }
		}
		
		// load weights
		distanceCalc.setObjectWeights(m_objWeights);

		int tempcount = 0;
		
		// iterate through scenes
		Iterator it = validateScenes.iterator();
		while( it.hasNext() )
		{
			//if (tempcount++ == 10) break;
			
			Scene thisScene = (Scene) it.next();

			totalScenes++;
			
			// record start time
			startTime = System.currentTimeMillis();
			
			Scene[] bestScenes = distanceCalc.findClosest(thisScene, storedScenes);
			
			// result is a set of candidate scenes stored in bestScenes; now, choose the best one to imitate
			//explain(bestScenes, thisScene, distanceCalc);
			
			// *********************************************
			// choose scene selection algorithm, parameters
			// *********************************************
			Scene matched;
			
			switch(m_sceneSel) {
			  case 1: {
				matched = SceneSelection.ChooseRandom(bestScenes, m_numBest);
				break;
			  }
			  case 2: {
				matched = SceneSelection.VoteWeighted(bestScenes, bestScenes.length, m_actionWeights);
				break;
			  }
			  case 3: {
				matched = SceneSelection.VoteWeightedWithRandom(bestScenes, bestScenes.length, m_actionWeights, 20);
				break;
			  }
			  default: {	// 0
				matched = SceneSelection.ChooseFirstValid(bestScenes);
				break;
			  }
			}
			
			// record end time
			endTime = System.currentTimeMillis();
			//System.out.println("Time taken: " + (endTime - startTime));
			totalTime += (endTime - startTime);

			// now compare the actions in thisScene with the actions in matched; track whether we got them right or not
			//System.out.println("Scene " + thisScene.getIdentString() + "/" + thisScene.getActions() +
			//                   " matches best with scene " + matched.getIdentString() + "/" + matched.getActions());
			                   			
			Vector knownActions = new Vector(thisScene.getActions());
			Iterator knownList = knownActions.iterator();
			while (knownList.hasNext())
			{			
				Action compareThis = (Action) knownList.next();
				
				if(compareThis.getAction() == Action.ACTION_KICK)
					origKick++;
				
				Vector actions = new Vector(matched.getActions());
				Iterator matchList = actions.iterator();
				while (matchList.hasNext())
				{
					Action doThis = (Action) matchList.next();
					
					if (doThis.getAction() == compareThis.getAction())
					{
						if(doThis.getAction() == Action.ACTION_KICK)
							correctKick++;

						// we have a matching action, so count it, and remove the items from the lists so they won't be counted twice
						totalMatches++;
						knownList.remove();
						matchList.remove();
						break;
					}
				}
				totalCompares++;
			}
		}
		
		// output statistics
		System.out.println("Validation complete.");
		System.out.println("Compared " + totalScenes + " scenes against database of " + storedScenes.size() + " scenes");
		System.out.println("Total action comparisons: " + (int)totalCompares);
		System.out.println("Total correct actions: " + (int)totalMatches + "(" + (totalMatches / totalCompares)*100 + "%)");
		System.out.println("Total KICK actions: " + (int)origKick);
		System.out.println("Total correct KICKs: " + (int)correctKick + "(" + (correctKick / origKick)*100 + "%)");
		System.out.println("Average time taken: " + totalTime / totalScenes + " mill. per scene");
		
		return;
	}


/**
	 * @param bestScenes
	 */
	private void explain(Scene[] bestScenes, Scene target, DistanceCalculation distcalc)
	{
		System.out.println("Best " + bestScenes.length + " matches to scene " + target.getIdentString());
		for(int i = 0; i < bestScenes.length; i++)
		{
			Scene x = bestScenes[i];
			System.out.println("   scene " + x.getIdentString() + ", distance " + distcalc.distanceBetween(x, target));		
		}
		System.out.println("Distance to self is " + distcalc.distanceBetween(target, target));				
		
	}

	//===========================================================================
// Private members
	private char					m_side;
	private	int						m_distCal;
	private int						m_sceneSel;
	private int						m_numBest;
	private float[]					m_objWeights;
	private float[]					m_actionWeights;
	
}
