/*
 * This class is based on Krislet's Brain and provides offline training/validation services to the KrisletScenes client.
 * It can be used as a testbed for statistically evaluating the performance of DistanceCalculation objects.
 * 
 * Created on Jan 26, 2005
 *
 */
package visiontable;

import java.util.*;
import java.io.*;

import sceneInfo.BallInfo;
import sceneInfo.Scene;
import sceneInfo.SceneLib;
import sceneInfo.VisualInfo;

/**
 * @author Kevin Lam
 *
 */
class ValidatorVT
{
	
	private ArrayList storedScenes = new ArrayList();
	private ArrayList validateScenes = new ArrayList();
	private SceneLib sceneLib = null;
	
	int numRows = 0;
	int numCols = 0;

	
	SceneStats gameStats;
	
	public ValidatorVT(String scenelib, String comparelib, int sceneSel, 
					int distCal, int numBest, float[] objWeights, float[] actionWeights)
	{
		m_side = 'l';	// arbitrary, shouldn't matter

		m_distCal = distCal;
		m_sceneSel = sceneSel;
		m_numBest = numBest;
		m_objWeights = objWeights;
		m_actionWeights = actionWeights;

		// initialize scene database
		System.out.println("Loading stored scene file " + scenelib);
		storedScenes = loadSceneFile(scenelib, storedScenes);
		
		int srows = numRows;
		int scols = numCols;
		
		if(comparelib.compareToIgnoreCase(scenelib) == 0)
		{
			System.out.println("Using the same scene file for target");
			validateScenes = storedScenes;
		}
		else
		{
			System.out.println("Loading target scene file " + comparelib);
			validateScenes = loadSceneFile(comparelib, validateScenes);
			System.out.println("Read " + validateScenes.size() + " scenes loaded for validation.");

		} 
   		
		//don't need to do this anymore, since we're not checking useing
		//cells anyway
   		// double check scene files match in dimension
		//if((srows != numRows)||(scols != numCols))
		//{
		//	System.err.println("Can't run validation - scene files differ in dimension");
		//}
		
		
   		//create a new stats object to keep track of the 
   		//number of scenes processed and time taken to process these
   		//scenes
   		gameStats = new SceneStats();
   		
	}

	// for a future version maybe try to encapsulate all these into a single utility class that other clients just hook into...
	// ... an item for refactoring later
	public ArrayList loadSceneFile(String fname, ArrayList target)
		{
			ArrayList loadedlist = null;
			System.out.println("Loading scenes from file " + fname);
			try
			{
				if (fname.endsWith(".lib")) {
					FileInputStream fileIn = new FileInputStream(fname);
					ObjectInputStream ois = new ObjectInputStream(fileIn);

					sceneLib = (SceneLib) ois.readObject();
					numRows = 0;
					numCols = 0;

					ois.close();
					fileIn.close();
				} else {
					FileInputStream fileIn = new FileInputStream(fname);
					ObjectInputStream ois = new ObjectInputStream(fileIn);
					
					numRows = ((Integer) ois.readObject()).intValue();
					numCols = ((Integer) ois.readObject()).intValue();
					
					//storedScenes = (Vector) ois.readObject();
					//target = (ArrayList) ois.readObject();
					//validateScenes = (ArrayList) ois.readObject();
					loadedlist = (ArrayList) ois.readObject();
					
					ois.close();
					fileIn.close();
				}
				
				/*FileInputStream fileIn = new FileInputStream(fname);
				
				ObjectInputStream ois = new ObjectInputStream(fileIn);
				numRows = ((Integer) ois.readObject()).intValue();
				numCols = ((Integer) ois.readObject()).intValue();
				target.addAll((ArrayList) ois.readObject());
				  
				ois.close();
				fileIn.close();
				*/
			}
			catch (FileNotFoundException e)
			{
				e.printStackTrace();
			}
			catch (IOException e)
			{
				e.printStackTrace();
			}
			catch (ClassNotFoundException e)
			{
				e.printStackTrace();
			}
		
			//System.out.println("Read " + target.size() + " scenes, table size (" + numRows + "," + numCols + ")");
			
			return loadedlist;
			
		}
		
	//  1. Get next Scene from target list
	//  3. Instantiate a DistanceCalculation object
	//  4. Pick out the best scene from the list
	//  5. Compare chosen action with stored action and get statistics
	//
	public void runValidation()
	{
		long startTime = 0, endTime = 0, totalTime = 0;
		float totalMatches = 0, totalCompares = 0;
		float origKick = 0, correctKick = 0;
		int totalScenes = 0;
		
		System.out.println("Beginning validation of " + validateScenes.size() + " scenes");
		
		DistanceCalculation distanceCalc;			// plug-and-play distance calculation object
				
		// *********************************************
		// choose scene selection algorithm, parameters
		// *********************************************
		switch(m_distCal) {
		  case 0: {
			  distanceCalc = new NearestNeighborCartesianObjects(m_numBest, m_side);
			  //distanceCalc = new NearestNeighborBallGoalDistance(m_numBest, m_side);
			  break;
			}
		  case 1: {
			distanceCalc = new NearestNeighborCartesianCellObjects(m_numBest, m_side);
			break;
		  }
		  case 2: {
			distanceCalc = new NearestNeighborCellBallGoalDistance(m_numBest, m_side);
			break;
		  }
		  case 3: {
			distanceCalc = new RandomDistance(m_numBest, m_side);
			break;
		  }
		  default: {
			distanceCalc = new NearestNeighborCartesianCellObjects(m_numBest, m_side);
			break;
		  }
		}
		
		// load weights
		distanceCalc.setObjectWeights(m_objWeights);

		int tempcount = 0;
		
		// iterate through scenes
		Iterator it = validateScenes.iterator();
		while( it.hasNext() )
		{
			//if (tempcount++ == 1000) break;
			
			Scene thisScene = (Scene) it.next();
			
			/////////////////////////////////////////////////////////////////
			//get the ball distance... only other change needed for the 
			//new scene lib to work correctly.. 
			VisualInfo stuff = thisScene.getVision();
			if (stuff.getBallList() == null) {
				stuff.setSceneDist(0.0f);
			} else {
				if (!stuff.getBallList().isEmpty()) {
					BallInfo tmpinfo = (BallInfo)stuff.getBallList().get(0);
					stuff.setSceneDist(tmpinfo.m_distance);
				} else {
					stuff.setSceneDist(0.0f);
				}
			}
			/////////////////////////////////////////////////////////////////
		
			totalScenes++;
			
			// record start time
			startTime = System.currentTimeMillis();
			
			////////////////////////////////////////////////////////////////////
			//if we're using a sceneLib library, get the set of scenes which 
			//correspond to the initial distance of the scene  - in this case
			//it is still the distance of the ball from the player
			//return this list of scenes as the old scene library.. this will
			//minimize the amount of code required to use this scenelib, and
			//*should* hide the fact that the library is chaning from everything
			//as well..
			if (sceneLib != null) {
				storedScenes = sceneLib.getSceneList(thisScene.getSceneDistance());
			}
			////////////////////////////////////////////////////////////////////
			
			
			Scene[] bestScenes = distanceCalc.findClosest(thisScene, storedScenes);
			
			// result is a set of candidate scenes stored in bestScenes; now, choose the best one to imitate
			//explain(bestScenes, thisScene, distanceCalc);
			
			// *********************************************
			// choose scene selection algorithm, parameters
			// *********************************************
			Scene matched;
			
			switch(m_sceneSel) {
			  case 1: {
				matched = SceneSelection.ChooseRandom(bestScenes, m_numBest);
				break;
			  }
			  case 2: {
				matched = SceneSelection.VoteWeighted(bestScenes, bestScenes.length, m_actionWeights);
				break;
			  }
			  case 3: {
				matched = SceneSelection.VoteWeightedWithRandom(bestScenes, bestScenes.length, m_actionWeights, 20);
				break;
			  }
			  default: {	// 0
				matched = SceneSelection.ChooseFirstValid(bestScenes);
				break;
			  }
			}
			
			// record end time
			endTime = System.currentTimeMillis();
			//System.out.println("Time taken: " + (endTime - startTime));
			totalTime += (endTime - startTime);

			// now compare the actions in thisScene with the actions in matched; track whether we got them right or not
			System.out.println("Scene " + thisScene.getIdentString() + "/" + thisScene.getActions() +
			                   " matches best with scene " + matched.getIdentString() + "/" + matched.getActions());
			                   			
			//record this cycles times
			//gameStats.addSceneInfo(distanceCalc.getSceneCount(), (endTime - startTime), matched.getIdentString(), bestScenes.length);

			
			Vector knownActions = new Vector(thisScene.getActions());
			Iterator knownList = knownActions.iterator();
			while (knownList.hasNext())
			{			
				Action compareThis = (Action) knownList.next();
				
				//if(compareThis.getAction() == Action.ACTION_KICK)
				//	origKick++;
				switch (compareThis.getAction()) {
					case Action.ACTION_NONE:
						break;
					case Action.ACTION_DASH:
						gameStats.addCorrectDASH();
						break;
					case Action.ACTION_KICK:
						gameStats.addCorrectKICK();
						break;
					case Action.ACTION_TURNNECK:
						gameStats.addCorrectTURNHEAD();
						break;
					case Action.ACTION_TURN:
						gameStats.addCorrectTURN();
						break;
					case Action.ACTION_CATCH:
						break;
					case Action.ACTION_MOVE:
						break;
				}

				Vector actions = new Vector(matched.getActions());
				Iterator matchList = actions.iterator();
				while (matchList.hasNext())
				{
					Action doThis = (Action) matchList.next();
					
					if (doThis.getAction() == compareThis.getAction())
					{
						//if(doThis.getAction() == Action.ACTION_KICK)
						//	correctKick++;

						//add checks for more actions and record them
						//to the scenestats object
						switch (doThis.getAction()) {
							case Action.ACTION_NONE:
								break;
							case Action.ACTION_DASH:
								gameStats.addMatchedDASH();
								break;
							case Action.ACTION_KICK:
								gameStats.addMatchedKICK();
								break;
							case Action.ACTION_TURNNECK:
								gameStats.addMatchedTURNHEAD();
								break;
							case Action.ACTION_TURN:
								gameStats.addMatchedTURN();
								break;
							case Action.ACTION_CATCH:
								break;
							case Action.ACTION_MOVE:
								break;
						}
						
						gameStats.addHit();
						
						// we have a matching action, so count it, and remove the items from the lists so they won't be counted twice
						totalMatches++;
						knownList.remove();
						matchList.remove();
						break;
					} else {
						//did not match, but will record what was selected in the
						//event of a miss
						switch (doThis.getAction()) {
							case Action.ACTION_NONE:
								break;
							case Action.ACTION_DASH:
								gameStats.addSelectedDASH();
								break;
							case Action.ACTION_KICK:
								gameStats.addSelectedKICK();
								break;
							case Action.ACTION_TURNNECK:
								gameStats.addSelectedTURNHEAD();
								break;
							case Action.ACTION_TURN:
								gameStats.addSelectedTURN();
								break;
							case Action.ACTION_CATCH:
								break;
							case Action.ACTION_MOVE:
								break;
						}			
						
						gameStats.addMiss();
						
					}
				}
				totalCompares++;
			}
		}
		
		// output statistics
		System.out.println("Validation complete.");
		System.out.println("Compared " + totalScenes + " scenes against database of " + storedScenes.size() + " scenes");
		System.out.println("Total action comparisons: " + (int)totalCompares);
		System.out.println("Total correct actions: " + (int)totalMatches + "(" + (totalMatches / totalCompares)*100 + "%)");
		System.out.println("Total KICK actions: " + (int)origKick);
		System.out.println("Total correct KICKs: " + (int)correctKick + "(" + (correctKick / origKick)*100 + "%)");
		System.out.println("Average time taken: " + totalTime / totalScenes + " mill. per scene");
		
		//when leaves the run, should be the end of the game - 
		//display the stats for this game
		gameStats.calcAvgs();
		System.out.println(gameStats.getStats());
		
		return;
	}


/**
	 * @param bestScenes
	 */
	private void explain(Scene[] bestScenes, Scene target, DistanceCalculation distcalc)
	{
		System.out.println("Best " + bestScenes.length + " matches to scene " + target.getIdentString());
		for(int i = 0; i < bestScenes.length; i++)
		{
			Scene x = bestScenes[i];
			System.out.println("   scene " + x.getIdentString() + ", distance " + distcalc.distanceBetween(x, target));		
		}
		System.out.println("Distance to self is " + distcalc.distanceBetween(target, target));				
		
	}

	//===========================================================================
// Private members
	private char					m_side;
	private	int						m_distCal;
	private int						m_sceneSel;
	private int						m_numBest;
	private float[]					m_objWeights;
	private float[]					m_actionWeights;
	
}
